function [pcOut] = fun_voxelize(pcIn, voxelDepth)
% Copyright (C) 2019 ECOLE POLYTECHNIQUE FEDERALE DE LAUSANNE, Switzerland
%
%     Multimedia Signal Processing Group (MMSPG)
%
%     This program is free software: you can redistribute it and/or modify
%     it under the terms of the GNU General Public License as published by
%     the Free Software Foundation, either version 3 of the License, or
%     (at your option) any later version.
%
%     This program is distributed in the hope that it will be useful,
%     but WITHOUT ANY WARRANTY; without even the implied warranty of
%     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%     GNU General Public License for more details.
%
%     You should have received a copy of the GNU General Public License
%     along with this program.  If not, see <http://www.gnu.org/licenses/>.
%
%
% Author:
%   Evangelos Alexiou (evangelos.alexiou@epfl.ch)
%
% Reference:
%   Alexiou, E., Viola, I., Borges, T., Fonseca, T., De Queiroz, R., &
%   Ebrahimi, T. (2019). A comprehensive study of the rate-distortion
%   performance in MPEG point cloud compression. APSIPA Transactions on
%   Signal and Information Processing, 8, E27.
%
%
% Quantization of a point cloud's coordinates to a regular grid of voxels.
%   The indices of the grid are non-negative integers that span in the range
%   [0, 2^voxelDepth-1]. The color assigned to each voxel is obtained after
%   sampling from colors of points that fall in the corresponding voxel.
%
%   [fileOut] = fun_voxelize(fileIn, voxelDepth)
%
%   INPUTS
%       fileIn:     A pointCloud object, or a path to a point cloud file.
%       voxelDepth: Non-negative integer that defines the depth of the
%                   voxel grid.
%
%   OUTPUTS
%       fileOut:    Voxelized point cloud.
%
%   Example:
%   [ptCloudOut] = fun_voxelize(ptCloudIn, 10)


if nargin < 2
    error('Too few input arguments.');
elseif nargin == 2
    if(any(ischar(pcIn)))
        ptCloud = pcread(pcIn);
    elseif(isa(pcIn,'pointCloud'))
        ptCloud = pcIn;
    end
end

% Vertices' coordinates
vertices = ptCloud.Location;

% Shift signal to fall in a range of positive numbers
vertices = vertices - min(vertices,[],1);

% Max of the range
maxRange = max(vertices(:));

% Quantization step
quantStep = maxRange/(2^(voxelDepth) - 1);

% Quantized vertices in the grid range [0, 2^N - 1]
quantVertices = floor(vertices/quantStep + 1/2);

% Unique quantized vertices as voxels
[quantVertices, I] = unique(quantVertices, 'rows');

% Unique position and color sampling of points in the same voxel
if isempty(ptCloud.Color) == 0
    pcOut = pointCloud(quantVertices, 'Color', ptCloud.Color(I,:));
else
    pcOut = pointCloud(quantVertices);
end
